package com.tencent.start.cgs.tools;

import org.apache.commons.codec.binary.Hex;

import java.io.*;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import static com.tencent.start.cgs.tools.ZpkFile.*;
import static com.tencent.start.cgs.tools.ZpkMetadata.*;

public class ZpkFileDiff extends ZpkFilePack {
    public static final String ZPKDIFF_EXT_NAME = ".zpkdiff";
    public static final String ZPKDIFF_FILENAME = ".cgs_vfs_diff";
    private final boolean strict;
    private final boolean list;
    private List<FileNode> listNew, listExists;

    private ZpkFile.Finder finder;
    private long dataPosOffset; // nfs local cache file need plus zpk file length
    private AbstractFile outputFile;

    public ZpkFileDiff(boolean strict, boolean list) throws IOException {
        this.strict = strict;
        this.list = list;
    }

    @Override
    protected long getDataOffset() throws IOException {
        long pos = output.getFilePointer();
        return dataPosOffset > 0 ? pos + dataPosOffset : pos;
    }

    @Override
    protected FileNode putIfAbsent(FileNode fileNode) {
        FileNode exists = super.putIfAbsent(fileNode);
        if (null == exists) {
            exists = finder.findChecksum(fileNode);
        }
        if (null != listNew) {
            (null == exists ? listNew : listExists).add(fileNode);
            fileNode.ud2 = exists;
        }
        return exists;
    }

    @Override
    protected void writeFileData(FileNode fileNode, AbstractFile file) throws IOException {
        FileNode exists = null;
        if (null != fileNode.md5sum) {
            exists = finder.findChecksum(fileNode);
        } else if (!strict) {
            exists = finder.find(fileNode);
        }
        if (null == exists) {
            super.writeFileData(fileNode, file);
            return;
        } else if (null != listExists) {
            listExists.add(fileNode);
            fileNode.ud2 = exists;
        }
        boolean s = null != fileNode.md5sum;
        fileNode.dataOffset = exists.dataOffset;
        fileNode.md5sum = exists.md5sum;
        traceTheSame(fileNode, exists, s);
    }

    private static void append(RandomAccessOutputFile out, InputStream in, final long length) throws IOException {
        int bytesRead, bytes;
        // appending tag
        final String appendingTag = getTimestampFileName(ZpkFile.APPEND_TAG, System.currentTimeMillis(), "");
        final long headPos = out.getFilePointer();
        final byte[] head = new byte[appendingTag.length()];
        if (length <= head.length || in.read(head) != head.length) {
            throw new IOException("head data length");
        }
        Speedometer sm = new Speedometer(length);
        out.write(appendingTag.getBytes(), 0, head.length);
        sm.feed(head.length);
        byte[] buffer = new byte[4 * 1024 * 1024];
        while ((bytes = (int) Math.min(buffer.length, sm.getRemainBytes())) > 0 &&
                (bytesRead = in.read(buffer, 0, bytes)) > 0) {
            out.write(buffer, 0, bytes);
            sm.feed(bytesRead);
        }
        out.seek(headPos);
        out.write(head);
        out.flush();
        if (0 != sm.getRemainBytes()) {
            throw new IOException("totalBytesRead != file.length()");
        }
        sm.finish();
    }

    @SuppressWarnings("ResultOfMethodCallIgnored")
    private RandomAccessOutputFile cache(final ZpkFile zpkFile) throws IOException {
        if (zpkFile.getFile().getClassType().equals(File.class)) {
            dataPosOffset = 0;
            return zpkFile.getOutput();
        }
        dataPosOffset = zpkFile.length();
        final File temp = File.createTempFile("~" + App.randomString(7), ZPKDIFF_EXT_NAME + ".tmp");
        temp.deleteOnExit();
        final RandomAccessOutputFile tempOut = RandomAccessOutputFile.from(temp);
        return new RandomAccessOutputFile.WrapperRandomAccessOutputFile(tempOut) {
            @Override
            public void close() throws IOException {
                super.close();
                try (FileInputStream in = new FileInputStream(temp);
                     RandomAccessOutputFile out = zpkFile.getOutput()) {
                    append(out, in, temp.length());
                } finally {
                    temp.delete();
                }
            } // close
        };
    }

    private void writeDiffEntry(ZpkFile zpkFile) throws IOException {
        try (RandomAccessByteArrayOutputStream out = new RandomAccessByteArrayOutputStream(1024)) {
            out.order(ByteOrder.LITTLE_ENDIAN);
            out.writeLong(zpkFile.length());
            out.write(zpkFile.getChecksum());
            String oldName = zpkFile.getFile().getName();
            String newName = getZpkFileName(getZpkPrefix(oldName), buildTime);
            byte[] newNameBytes = newName.getBytes(StandardCharsets.UTF_8);
            byte[] oldNameBytes = oldName.getBytes(StandardCharsets.UTF_8);
            if (newNameBytes.length > MAX_ZPK_FILE_NAME_LEN ||
                    oldNameBytes.length > MAX_ZPK_FILE_NAME_LEN) {
                throw new IOException("file name too long: " + newName + ", " + oldName);
            }
            out.writeByte(newNameBytes.length);
            out.write(newNameBytes);
            out.writeByte(oldNameBytes.length);
            out.write(oldNameBytes);
            byte[] data = out.toByteArray();
            crc32.reset();
            crc32.update(data, 0, data.length);
            writeEntry(output, ZPKDIFF_FILENAME, data.length, buildTime, crc32.getValue(), false);
            output.write(data);
        }
    }

    private void writeList(AbstractFile file) throws IOException {
        try (PrintStream out = new PrintStream(RandomAccessOutputFile.from(file))) {
            long totalSize = 0;
            for (FileNode fileNode : listNew) {
                totalSize += fileNode.fileSize;
                out.println("update: " + fileNode.relativeName + ", size: " +
                        humanReadableBytes(fileNode.fileSize) + ", mtime: " +
                        sdf1.format(fileNode.lastModified) + ", md5: " +
                        Hex.encodeHexString(fileNode.md5sum));
            }
            out.println();
            out.println("" + listNew.size() + " files updated. " + humanReadableBytes(totalSize));
            out.println();
            totalSize = 0;
            for (FileNode fileNode : listExists) {
                totalSize += fileNode.fileSize;
                FileNode exists = (FileNode) fileNode.ud2;
                out.println("not modified: " + fileNode.relativeName +
                        " (" + sdf1.format(fileNode.lastModified) + ") -> " +
                        exists.relativeName + " (" + sdf1.format(exists.lastModified) + "), size: " +
                        humanReadableBytes(fileNode.fileSize) + ", md5: " +
                        Hex.encodeHexString(exists.md5sum));
            }
            out.println();
            out.print("" + listExists.size() + " files not modified. " + humanReadableBytes(totalSize));
        }
    }

    public long diff(AbstractFile outputDir, AbstractFile zFile, AbstractFile inputDir) throws IOException {
        if (null == outputDir) {
            outputDir = zFile.getParentFile();
        }
        if (this.list) {
            listNew = new ArrayList<>(128 * 1024);
            listExists = new ArrayList<>(128 * 1024);
        }
        String name = zFile.getName();
        String prefix = getZpkPrefix(name);
        String ts = getZpkTimestamp(name);
        String diffFileName = getTimestampFileName(prefix + "_" + ts + "_to", buildTime, ".zpkdiff");
        AbstractFile outputFile = outputDir.getChildFile(diffFileName);
        System.out.println("Output File: " + outputFile);
        this.outputFile = outputFile;
        AbstractFile tempFile = getTempFile(outputFile);
        long bytesWritten;
        try (ZpkFile zpkFile = ZpkFile.open(zFile);
             RandomAccessOutputFile out = RandomAccessOutputFile.from(tempFile)) {
            if (buildTime < zpkFile.getLastModified()) {
                throw new IOException("time error");
            }
            out.order(ByteOrder.LITTLE_ENDIAN);
            final long startPos = out.getFilePointer();
            output = out;
            finder = zpkFile.finder();
            writeDiffEntry(zpkFile);
            if (out.getFilePointer() > zpkFile.length) {
                throw new IOException("invalid data position offset");
            }
            dataPosOffset = zpkFile.length - out.getFilePointer();
            writeDataSection(inputDir);
            finder = null;
            output = null;
            bytesWritten = out.length() - startPos;
        } catch (Exception e) {
            tempFile.delete();
            throw e;
        }
        if (outputFile.exists() && !outputFile.delete()) {
            tempFile.delete();
            throw new IOException("count not delete file: " + outputFile);
        } else if (!tempFile.renameTo(outputFile)) {
            throw new IOException("count not rename: " + outputFile);
        }
        if (null != this.listNew) {
            writeList(outputDir.getChildFile(diffFileName + ".list"));
        }
        return bytesWritten;
    }

    private long patch(AbstractFile outputFile, RandomAccessInputFile in, ZpkFile zpkFile) throws IOException {
        final long dataEntryPos = in.getFilePointer();
        ZpkEntry dataEntry = readEntry(in);
        if (!dataEntry.name.startsWith(DATA_FILENAME)) {
            throw new IOException("not valid diff file: " + zpkFile.getFile());
        }
        final long dataEntrySize = in.getFilePointer() - dataEntryPos;
        System.out.println("Output File: " + outputFile);
        this.outputFile = outputFile;
        AbstractFile tempFile = getTempFile(outputFile);
        tempFile.link(zpkFile.getFile());
        long bytesWritten;
        try (RandomAccessOutputFile out = RandomAccessOutputFile.from(tempFile)) {
            out.order(ByteOrder.LITTLE_ENDIAN);
            out.seek(zpkFile.length);
            output = out;
            in.seek(dataEntryPos);
            bytesWritten = dataEntrySize + dataEntry.size;
            append(out, in, bytesWritten);
            output = null;
        } catch (Exception e) {
            tempFile.delete();
            throw e;
        }
        if (outputFile.exists() && !outputFile.delete()) {
            tempFile.delete();
            throw new IOException("count not delete file: " + outputFile);
        } else if (!tempFile.renameTo(outputFile)) {
            throw new IOException("count not rename: " + outputFile);
        }
        return bytesWritten;
    }

    public long patch(AbstractFile outputDir, AbstractFile zFile, AbstractFile inputFile) throws IOException {
        try (RandomAccessInputFile in = RandomAccessInputFile.from(inputFile);
             ZpkFile zpkFile = ZpkFile.open(zFile)) {
            ZpkEntry headEntry = readEntry(in);
            if (!ZPKDIFF_FILENAME.equals(headEntry.name) || headEntry.size < 62 || headEntry.size > 512) {
                throw new IOException("not valid diff file: " + inputFile);
            }
            in.order(ByteOrder.LITTLE_ENDIAN);
            final long length = in.readLong();
            final byte[] checksum = new byte[16];
            in.readFully(checksum);
            final int newNameLength = in.read();
            if (newNameLength > MAX_ZPK_FILE_NAME_LEN) {
                throw new IOException("invalid name length: " + newNameLength);
            }
            final String newZpkFileName = new String(in.readBytes(newNameLength), StandardCharsets.UTF_8);
            final int oldNameLength = in.read();
            if (oldNameLength > MAX_ZPK_FILE_NAME_LEN) {
                throw new IOException("invalid name length: " + oldNameLength);
            }
            final String oldZpkFileName = new String(in.readBytes(oldNameLength), StandardCharsets.UTF_8);
            if (zpkFile.length != length || !Arrays.equals(zpkFile.getChecksum(), checksum)) {
                throw new IOException("zpk file not match. expected " + oldZpkFileName);
            } else if (!oldZpkFileName.equals(zFile.getName())) {
                System.out.println("WARING: zpk file name not equals. expected \"" + oldZpkFileName +
                        "\" but get \"" + zFile.getName() + "\".");
            }
            if (null == outputDir) {
                outputDir = zFile.getParentFile();
            }
            return patch(outputDir.getChildFile(newZpkFileName), in, zpkFile);
        } // try
    }

    public long update(AbstractFile outputDir, AbstractFile zFile, AbstractFile inputDir) throws IOException {
        if (null == outputDir) {
            outputDir = zFile.getParentFile();
        }
        if (this.list) {
            listNew = new ArrayList<>(128 * 1024);
            listExists = new ArrayList<>(128 * 1024);
        }
        String name = zFile.getName();
        String nameWithTimestamp = getZpkFileName(getZpkPrefix(name), buildTime);
        AbstractFile outputFile = outputDir.getChildFile(nameWithTimestamp);
        this.outputFile = outputFile;
        System.out.println("Output File: " + outputFile);
        AbstractFile tempFile = getTempFile(outputFile);
        tempFile.link(zFile);
        long bytesWritten;
        try (ZpkFile zpkFile = ZpkFile.open(tempFile);
             RandomAccessOutputFile out = cache(zpkFile)) {
            if (buildTime < zpkFile.getLastModified()) {
                throw new IOException("time error");
            }
            out.order(ByteOrder.LITTLE_ENDIAN);
            final long startPos = out.getFilePointer();
            output = out;
            finder = zpkFile.finder();
            writeDataSection(inputDir);
            finder = null;
            output = null;
            bytesWritten = out.length() - startPos;
        } catch (Exception e) {
            tempFile.delete();
            throw e;
        }
        if (outputFile.exists() && !outputFile.delete()) {
            tempFile.delete();
            throw new IOException("count not delete file: " + outputFile);
        } else if (!tempFile.renameTo(outputFile)) {
            throw new IOException("count not rename: " + outputFile);
        }
        if (null != this.listNew) {
            writeList(outputDir.getChildFile(nameWithTimestamp + ".list"));
        }
        return bytesWritten;
    }

    public static void printUsage() {
        System.out.println("  java -jar gamepack.jar zpk update -z <zpk_file> -i <input_dir> [-o <output_dir>] [-s] [-l]");
        System.out.println("  java -jar gamepack.jar zpk diff -z <zpk_file> -i <input_dir> [-o <output_dir>] [-s] [-l]");
        System.out.println("  java -jar gamepack.jar zpk patch -z <zpk_file> -i <input_diff> [-o <output_dir>]");
    }

    public static void usage() {
        System.out.println("Usage:");
        printUsage();
        System.exit(1);
    }

    public static void callMain(String[] args) throws IOException {
        boolean strict = false;
        boolean list = false;
        AbstractFile zFile = null;
        AbstractFile inputFile = null;
        AbstractFile outputFile = null;
        if (args.length < 6) {
            usage();
            return;
        }
        final String op = args[1];
        int i = 2;
        while (i < args.length) {
            switch (args[i++]) {
                case "-z":
                    if (i < args.length) {
                        zFile = AbstractFile.make(args[i++]);
                    }
                    break;
                case "-i":
                    if (i < args.length) {
                        inputFile = AbstractFile.make(args[i++]);
                    }
                    break;
                case "-o":
                    if (i < args.length) {
                        outputFile = AbstractFile.make(args[i++]);
                    }
                    break;
                case "-d":
                    ENABLE_DEBUG = true;
                    break;
                case "-s":
                    strict = true;
                    break;
                case "-l":
                    list = true;
                    break;
                default:
                    usage();
                    break;
            }
        }
        if (null == zFile || null == inputFile) {
            usage();
            return;
        }
        String name = zFile.getName();
        if (name.length() < 21) {
            throw new IOException("invalid zpk file name");
        }
        ZpkFileDiff builder = new ZpkFileDiff(strict, list);
        final long buildTime = builder.buildTime;
        long bytesWritten = 0;
        switch (op) {
            case "update":
                bytesWritten = builder.update(outputFile, zFile, inputFile);
                break;
            case "diff":
                bytesWritten = builder.diff(outputFile, zFile, inputFile);
                break;
            case "patch":
                bytesWritten = builder.patch(outputFile, zFile, inputFile);
                break;
            default:
                usage();
                break;
        }
        outputFile = builder.outputFile;
        final long cost = System.currentTimeMillis() - buildTime;
        System.out.println("Done! ");
        System.out.println("Output File: " + outputFile + ". "
                + "Size: " + humanReadableBytes(bytesWritten) + ". "
                + "Speed: " + humanReadableBytes(1000.0 * bytesWritten / cost) + "/s" + ". "
                + "Cost: " + humanReadableTime(cost));
        System.out.println();
        System.exit(0);
    } // callMain
} // class ZpkFileDiff
