package com.tencent.start.cgs.tools;

import org.apache.commons.codec.binary.Hex;

import java.io.*;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.zip.DataFormatException;
import java.util.zip.Deflater;
import java.util.zip.Inflater;

import static com.tencent.start.cgs.tools.App.*;
import static com.tencent.start.cgs.tools.ZpkMetadata.FileNode;

@SuppressWarnings("unused")
public class ZpkFile implements Closeable {
    public static final String EXT_NAME = ".zpk";
    public static final String HEADER_FILENAME = ".cgs_vfs_cgpk_header";
    public static final String DATA_FILENAME = ".cgs_vfs_data";
    public static final String APPEND_TAG = ".cgs_vfs_appending_tag_eOZXEYNh0";
    public static final int ZIP_ENTRY_SIZE = 30;
    public static final int ZIP_ENTRY_SIZE_64 = 50;
    public static final String DATA_FOOTER_MAGIC = "PKFF";

    public static final int MAX_ZPK_FILE_NAME_LEN = 127;

    // @SuppressWarnings("SameParameterValue")
    public static String getTimestampFileName(String prefix, long time, String extName) {
        return prefix + "_" + new SimpleDateFormat("yyyyMMdd_HHmmss").format(time) + extName;
    }

    public static String getZpkPrefix(String name) {
        return name.substring(0, name.length() - 16 - ZpkFile.EXT_NAME.length());
    }

    public static String getZpkTimestamp(String name) {
        return name.substring(name.length() - EXT_NAME.length() - 15, name.length() - EXT_NAME.length());
    }

    public static String getZpkFileName(String prefix, long time) {
        return getTimestampFileName(prefix, time, EXT_NAME);
    }

    public static boolean isValidZpkFileName(String name) {
        return !(name.length() < 20 || !name.substring(name.length() - 4).equalsIgnoreCase(EXT_NAME));
    }

    @Override
    public void close() throws IOException {
        if (input != null) {
            input.close();
        }
    }

    public static class ZpkEntry {
        public final String name;
        public final long size;
        public final long time;
        public final long crc32;

        public ZpkEntry(String name, long size, long time, long crc32) {
            this.name = name;
            this.size = size;
            this.time = time;
            this.crc32 = crc32;
        }
    }

    @SuppressWarnings("SameParameterValue")
    public static void writeEntry(DataOutput output, String name, long size, long time, long crc32,
                                  boolean forceSize64) throws IOException {
        if (!forceSize64 && size >= SIZE_4G) {
            forceSize64 = true;
        }
        // 4 Local file header signature = 0x04034b50
        output.writeInt(0x04034b50);
        // 2 Version needed to extract (minimum)
        output.writeShort((short) (forceSize64 ? 45 : name.endsWith("/") ? 20 : 10));
        // 2 General purpose bit flag
        output.writeShort((short) 0);
        // 2 Compression method; e.g. none = 0, DEFLATE = 8
        output.writeShort((short) 0);
        // 4 File last modification time
        output.writeInt(javaToDosTime(time));
        // 4 CRC-32 of uncompressed data
        output.writeInt((int) crc32);
        // 4 Compressed size (or 0xffffffff for ZIP64)
        output.writeInt(forceSize64 ? 0xffffffff : (int) size);
        // 4 Uncompressed size (or 0xffffffff for ZIP64)
        output.writeInt(forceSize64 ? 0xffffffff : (int) size);
        byte[] nameBytes = name.getBytes(StandardCharsets.UTF_8);
        // 2 File name length (n)
        output.writeShort((short) nameBytes.length);
        // 2 Extra data length (n)
        output.writeShort(forceSize64 ? (short) 20 : (short) 0);
        output.write(nameBytes);
        if (forceSize64) {
            output.writeShort((short) 1);
            output.writeShort((short) 16);
            output.writeLong(size);
            output.writeLong(size);
        }
    }

    public static void compressEntry(DataOutput output, String name, long time, byte[] data, int off, int size)
            throws IOException {
        int compressedSize = size;
        CRC32 crc32 = new CRC32();
        crc32.update(data, off, size);
        if (size > 128) {
            try {
                Deflater def = new Deflater(Deflater.DEFAULT_COMPRESSION, true);
                def.setInput(data, off, size);
                def.finish();
                byte[] outBuf = new byte[size];
                int len = def.deflate(outBuf, 0, outBuf.length);
                def.end();
                if (len < size) {
                    compressedSize = len;
                    data = outBuf;
                    off = 0;
                }
            } catch (Exception ignored) {
            }
        }
        // 4 Local file header signature = 0x04034b50
        output.writeInt(0x04034b50);
        // 2 Version needed to extract (minimum)
        output.writeShort((short) (size == compressedSize ? 10 : 20));
        // 2 General purpose bit flag
        output.writeShort((short) 0x800);
        // 2 Compression method; e.g. none = 0, DEFLATE = 8
        output.writeShort(compressedSize == size ? (short) 0 : (short) 8);
        // 4 File last modification time
        output.writeInt(javaToDosTime(time));
        // 4 CRC-32 of uncompressed data
        output.writeInt((int) crc32.getValue());
        // 4 Compressed size (or 0xffffffff for ZIP64)
        output.writeInt(compressedSize);
        // 4 Uncompressed size (or 0xffffffff for ZIP64)
        output.writeInt(size);
        byte[] nameBytes = name.getBytes(StandardCharsets.UTF_8);
        // 2 File name length (n)
        output.writeShort((short) nameBytes.length);
        // 2 Extra data length (n)
        output.writeShort((short) 0);
        output.write(nameBytes);
        output.write(data, off, compressedSize);
        /*
        if (compressedSize != size) {
            try {
                Inflater inf = new Inflater(true);
                inf.setInput(data, off, compressedSize);
                byte[] outBuf = new byte[size];
                int bytes = inf.inflate(outBuf);
                if (bytes != size) {
                    throw new IOException("");
                }
                inf.end();
            } catch (Exception ignored) {
            }
        }
         */
    }

    public static void compressEntry(DataOutput output, String name, long time, byte[] data)
            throws IOException {
        compressEntry(output, name, time, data, 0, data.length);
    }

    public static Map.Entry<String, byte[]> uncompressEntry(RandomAccessDataInput input) throws IOException {
        input.order(ByteOrder.LITTLE_ENDIAN);
        // 4 Local file header signature = 0x04034b50
        if (0x04034b50 != input.readInt()) {
            throw new IOException("bad format: zip entry header signature");
        }
        // 2 Version needed to extract (minimum)
        short ver = input.readShort();
        if (10 != ver && 20 != ver) {
            throw new IOException("bad format: zip entry version");
        }
        // 2 General purpose bit flag
        input.readShort();
        // 2 Compression method; e.g. none = 0, DEFLATE = 8
        short method = input.readShort();
        if (0 != method && 8 != method) {
            throw new IOException("bad format: compression method");
        }
        // 4 File last modification time
        long lastModifiedDosTime = input.readUnsignedInt();
        // 4 CRC-32 of uncompressed data
        long crc32 = input.readUnsignedInt();
        // 4 Compressed size (or 0xffffffff for ZIP64)
        int compressedSize = (int) input.readUnsignedInt();
        // 4 Uncompressed size (or 0xffffffff for ZIP64)
        int uncompressedSize = (int) input.readUnsignedInt();
        // 2 File name length (n)
        int nameLength = input.readUnsignedShort();
        if (nameLength > 255) {
            throw new IOException("bad format: name length");
        }
        // 2 Extra data length (n)
        int extraDataLength = input.readUnsignedShort();
        if (0 != extraDataLength) {
            throw new IOException("bad format: extra data length");
        }
        // name
        byte[] nameBytes = new byte[nameLength];
        input.readFully(nameBytes);

        byte[] inBuf = new byte[compressedSize];
        input.readFully(inBuf);

        final byte[] outBuf = new byte[uncompressedSize];
        try {
            Inflater inf = new Inflater(true);
            inf.setInput(inBuf);
            int bytes = inf.inflate(outBuf);
            if (bytes != uncompressedSize) {
                throw new IOException("bad size: bytes=" + bytes + ", uncompressedSize=" + uncompressedSize);
            }
            inf.end();
            final String name = new String(nameBytes, StandardCharsets.UTF_8);
            return new Map.Entry<>() {
                @Override
                public String getKey() {
                    return name;
                }
                @Override
                public byte[] getValue() {
                    return outBuf;
                }
                @Override
                public byte[] setValue(byte[] value) {
                    throw new UnsupportedOperationException();
                }
            };
        } catch (DataFormatException e) {
            throw new IOException(e);
        }

    }

    public static ZpkEntry readEntry(RandomAccessDataInput input) throws IOException {
        input.order(ByteOrder.LITTLE_ENDIAN);
        // 4 Local file header signature = 0x04034b50
        if (0x04034b50 != input.readInt()) {
            throw new IOException("bad format: zip entry header signature");
        }
        // 2 Version needed to extract (minimum)
        short ver = input.readShort();
        if (45 != ver && 20 != ver && 10 != ver) {
            throw new IOException("bad format: zip entry version");
        }
        // 2 General purpose bit flag
        input.readShort();
        // 2 Compression method; e.g. none = 0, DEFLATE = 8
        if (0 != input.readShort()) {
            throw new IOException("bad format: compression method");
        }
        // 4 File last modification time
        long lastModifiedDosTime = input.readUnsignedInt();
        // 4 CRC-32 of uncompressed data
        long crc32 = input.readUnsignedInt();
        // 4 Compressed size (or 0xffffffff for ZIP64)
        long compressedSize = input.readUnsignedInt();
        // 4 Uncompressed size (or 0xffffffff for ZIP64)
        long uncompressedSize = input.readUnsignedInt();
        if (compressedSize != uncompressedSize) {
            throw new IOException("bad format: compressed/uncompressed size");
        }
        // 2 File name length (n)
        int nameLength = input.readUnsignedShort();
        if (nameLength > 255) {
            throw new IOException("bad format: name length");
        }
        // 2 Extra data length (n)
        int extraDataLength = input.readUnsignedShort();
        if (0 != extraDataLength && 20 != extraDataLength) {
            throw new IOException("bad format: extra data length");
        } else if (20 == extraDataLength && 0xFFFFFFFFL != compressedSize) {
            throw new IOException("bad format: extra data length/compressed size");
        }
        // name
        byte[] nameBytes = new byte[nameLength];
        input.readFully(nameBytes);
        if (extraDataLength > 0) {
            if (1 != input.readShort() || 16 != input.readShort()) {
                throw new IOException("bad format: extra data");
            }
            compressedSize = input.readLong();
            uncompressedSize = input.readLong();
            if (compressedSize != uncompressedSize) {
                throw new IOException("bad format: extra data compressed/uncompressed size");
            }
        }
        return new ZpkEntry(new String(nameBytes, StandardCharsets.UTF_8), uncompressedSize,
                App.dosToJavaTime(lastModifiedDosTime), crc32);
    }

    public static boolean checkAppendingTag(RandomAccessDataInput input) throws IOException {
        byte[] buf = new byte[APPEND_TAG.length()];
        input.readFully(buf);
        return APPEND_TAG.equals(new String(buf));
    }

    /*
    Data Section Footer Format:
        4b MAGIC:PKFF
        8b metadata offset
        4b metadata size
        Nb padding
        1b footer size

    File Metadata Header Format:
        4b MAGIC:PKFL
        1b MajorVer
        1b MinorVer
        2b byte order check
        1b Header size
        1b Node size
        4b Nodes offset
        4b Nodes count
        2b padding

    Node format:
        1b High 4bits flags (0x80 directory), low 4bits nameLengthHigh
        struct {
            1b nameLengthLow
            4b name ptr
        }
        2b lastModifiedHigh
        4b lastModifiedLow
        8b fileSize
        union {
            8b dataOffset
            struct {
                4b childNodesCount
                4b childNodesListPtr
            }
        }
     */
    public class Version {
        public final String version;
        public final long offset;
        public final long length;
        private Reader reader;

        private Version(String version, long offset, long length) {
            this.version = version;
            this.offset = offset;
            this.length = length;
        }

        public Reader reader() throws IOException {
            if (null == reader) {
                reader = new Reader();
            }
            return reader;
        }

        public class Reader {
            public final long metadataPos;
            public final int metadataLen;
            public final FileNode root;

            private Reader() throws IOException {
                input.seek(offset + length - 1);
                final int footerSize = input.readByte();
                byte[] magic = new byte[4];
                input.seek(offset + length - 1 - footerSize);
                input.readFully(magic);
                if (!DATA_FOOTER_MAGIC.equals(new String(magic))) {
                    throw new IOException("data section footer magic");
                }
                metadataPos = input.readLong();
                final long metadataSize = input.readUnsignedInt();
                final long metadataTotalSize = input.readUnsignedInt();
                if (metadataSize > metadataTotalSize || metadataTotalSize > 0x7FFFFFFFL || metadataTotalSize > length) {
                    throw new IOException("file metadata size");
                }
                metadataLen = (int) metadataTotalSize;
                root = new ZpkMetadata.Loader(input, offset + metadataPos, metadataLen).load();
            }

            private void dumpFile(OutputStream output, long offset, long fileSize, MessageDigest md)
                    throws IOException {
                input.seek(offset);
                long remainBytes;
                Speedometer speedometer = new Speedometer(fileSize);
                byte[] buffer = new byte[4 * 1024 * 1024];
                while ((remainBytes = speedometer.getRemainBytes()) > 0) {
                    int bytesRead = (int) Math.min(buffer.length, remainBytes);
                    input.readFully(buffer, 0, bytesRead);
                    output.write(buffer, 0, bytesRead);
                    if (md != null) {
                        md.update(buffer, 0, bytesRead);
                    }
                    speedometer.feed(bytesRead);
                }
                if (0 != speedometer.getRemainBytes()) {
                    throw new IOException("totalBytesRead != file.length()");
                }
                speedometer.finish();
            }

            MessageDigest md5;

            MessageDigest md() throws IOException {
                if (null == md5) {
                    try {
                        md5 = MessageDigest.getInstance("MD5");
                    } catch (NoSuchAlgorithmException e) {
                        throw new IOException(e);
                    }
                }
                return md5;
            }

            @SuppressWarnings("ResultOfMethodCallIgnored")
            private void dumpFile(File file, FileNode fileNode, String name) throws IOException {
                MessageDigest md = md();
                md.reset();
                try (OutputStream out = Files.newOutputStream(file.toPath())) {
                    System.out.println("Write file: " + name + ", Size: " + humanReadableBytes(fileNode.fileSize));
                    File parentFile = file.getParentFile();
                    if (!parentFile.exists()) {
                        parentFile.mkdirs();
                    }
                    dumpFile(out, fileNode.dataOffset, fileNode.fileSize, md);
                }
                if (fileNode.fileSize > 0) {
                    byte[] md5sum = md.digest();
                    if (!Arrays.equals(fileNode.md5sum, md5sum)) {
                        throw new IOException("md5 check failed: " +
                                fileNode.relativeName + ". expected " +
                                Hex.encodeHexString(fileNode.md5sum) + " but " +
                                Hex.encodeHexString(md5sum) + ". size: " + fileNode.fileSize);
                    }
                }
            }

            @SuppressWarnings("ResultOfMethodCallIgnored")
            private void recurseDump(File output, FileNode fileNode, String relativePath) throws IOException {
                if (null == fileNode.childNodes) {
                    throw new IOException("not a directory: " + output.getPath());
                } else if (!output.exists() && !output.mkdirs()) {
                    throw new IOException("count not create directory: " + output.getPath());
                }
                for (FileNode childNode : fileNode.childNodes) {
                    File file = new File(output, childNode.fileName);
                    String relativeName = relativePath + childNode.fileName;
                    if (null != childNode.childNodes) {
                        recurseDump(file, childNode, relativeName + "/");
                    } else {
                        dumpFile(file, childNode, relativeName);
                    }
                    file.setLastModified(fileNode.lastModified);
                }
            }

            public long dump(File output) throws IOException {
                recurseDump(output, root, "");
                return root.fileSize;
            }

            public FileNode getNode(String path) throws IOException {
                int n = 0;
                final int len = path.length();
                while (n < len && path.charAt(n) == '/') {
                    ++n;
                }
                FileNode parent = root;
                while (n < len && null != parent.childNodes) {
                    int m = path.indexOf('/', n);
                    String name = -1 == m ? path.substring(n) : path.substring(n, m);
                    int index = Collections.binarySearch(parent.childNodes, new FileNode(name));
                    if (index < 0) {
                        break;
                    }
                    FileNode node = parent.childNodes.get(index);
                    if (-1 == m) {
                        return node;
                    }
                    parent = node;
                    n = m + 1;
                }
                return null;
            }

            public RandomAccessInputFile open(final FileNode node) throws IOException {
                return new RandomAccessInputFile() {
                    final RandomAccessInputFile in = RandomAccessInputFile.from(file);
                    long pos = 0;
                    @Override
                    public void close() throws IOException {
                        in.close();
                    }

                    @Override
                    public long length() {
                        return node.fileSize;
                    }

                    @Override
                    public long getFilePointer() {
                        return this.pos;
                    }

                    @Override
                    public void seek(long pos) {
                        this.pos = pos;
                    }

                    @Override
                    public void readFully(byte[] buf, int off, int len) throws IOException {
                        if (buf == null) {
                            throw new NullPointerException("buf == null");
                        } else if (off < 0 || len < 0 || off + len > buf.length) {
                            throw new IndexOutOfBoundsException();
                        } else if (len == 0) {
                            return;
                        } else if (this.pos >= node.fileSize) {
                            throw new EOFException();
                        }
                        int size = (int) Math.min(len, node.fileSize - this.pos);
                        this.in.seek(node.dataOffset + this.pos);
                        this.in.readFully(buf, off, size);
                        this.pos += size;
                        if (size < len) {
                            throw new EOFException();
                        }
                    }

                    @Override
                    public int skipBytes(int n) {
                        if (this.pos >= node.fileSize) {
                            return 0;
                        }
                        int size = (int) Math.min(n, node.fileSize - this.pos);
                        this.pos += size;
                        return size;
                    }

                    @Override
                    public int read() throws IOException {
                        if (pos >= node.fileSize) {
                            return -1;
                        }
                        ++pos;
                        return this.in.read();
                    }

                    @Override
                    public int pread(byte[] buf, int off, int len, long position) throws IOException {
                        if (off < 0 || len < 0) {
                            throw new IndexOutOfBoundsException();
                        } else if (position >= length()) {
                            return -1;
                        } else if (position + len > length()) {
                            len = (int) (length() - position);
                        }
                        return in.pread(buf, off, len, position);
                    }
                };
            }

            public RandomAccessInputFile open(String path) throws IOException {
                FileNode node = getNode(path);
                if (null == node) {
                    throw new FileNotFoundException(path);
                }
                return open(node);
            }
        } // class Reader
    } // class Version

    static final Comparator<FileNode> c = (o1, o2) -> {
        if (o1.fileSize < o2.fileSize) {
            return -1;
        } else if (o1.fileSize > o2.fileSize) {
            return 1;
        } else if (o1.lastModified < o2.lastModified) {
            return -1;
        } else if (o1.lastModified > o2.lastModified) {
            return 1;
        }
        return o1.relativeName.compareToIgnoreCase(o2.relativeName);
    };

    public static class Finder {
        private final Map<String, FileNode> map;
        private final FileNode[] array;

        public Finder(List<Version> versions) throws IOException {
            int size = 0;
            for (Version version : versions) {
                size += version.reader().root.totalChildNoesCount;
            }
            map = new HashMap<>(size);
            ArrayList<FileNode> nodes = new ArrayList<>(size);
            for (Version version : versions) {
                recurseAdd(version.reader().root, "", nodes);
            }
            array = nodes.toArray(new FileNode[0]);
            Arrays.sort(array, c);
        }

        public Finder(Version version) throws IOException {
            int size = version.reader().root.totalChildNoesCount;
            map = new HashMap<>(size);
            ArrayList<FileNode> nodes = new ArrayList<>(size);
            recurseAdd(version.reader().root, "", nodes);
            array = nodes.toArray(new FileNode[0]);
            Arrays.sort(array, c);
        }

        public Finder(RandomAccessDataInput in, long off, int len) throws IOException {
            FileNode root = new ZpkMetadata.Loader(in, off, len).load();
            int size = root.totalChildNoesCount;
            map = new HashMap<>(size);
            ArrayList<FileNode> nodes = new ArrayList<>(size);
            recurseAdd(root, "", nodes);
            array = nodes.toArray(new FileNode[0]);
            Arrays.sort(array, c);

        }

        private void recurseAdd(FileNode fileNode, String relativePath, ArrayList<FileNode> nodes) {
            if (null == fileNode.childNodes) {
                return;
            }
            for (FileNode childNode : fileNode.childNodes) {
                childNode.relativeName = relativePath + childNode.fileName;
                if (childNode.isDirectory()) {
                    recurseAdd(childNode, childNode.relativeName + "/", nodes);
                } else if (childNode.fileSize > 0) {
                    String key = Hex.encodeHexString(childNode.md5sum) + childNode.fileSize;
                    if (null == map.putIfAbsent(key, childNode)) {
                        nodes.add(childNode);
                    }
                }
            }
        }

        public FileNode find(FileNode fileNode) {
            int n = Arrays.binarySearch(array, fileNode, c);
            if (n < 0) {
                return null;
            }
            FileNode exists = array[n];
            return (exists.fileSize == fileNode.fileSize &&
                    exists.lastModified == fileNode.lastModified &&
                    exists.relativeName.equalsIgnoreCase(fileNode.relativeName)) ? exists : null;
        }

        public FileNode findChecksum(FileNode fileNode) {
            return map.get(Hex.encodeHexString(fileNode.md5sum) + fileNode.fileSize);
        }
    }
    protected final AbstractFile file;
    protected final RandomAccessInputFile input;
    protected final long length;
    protected final long lastModified;
    protected final byte[] checksum;
    protected final List<Version> versions = new ArrayList<>();
    protected Finder finder;

    protected static ZpkEntry loadEntry(RandomAccessDataInput input, DataOutput mem) throws IOException {
        ZpkEntry entry = readEntry(input);
        mem.writeBytes(entry.name);
        mem.writeLong(entry.time);
        mem.writeLong(entry.size);
        mem.writeInt((int) entry.crc32);
        return entry;
    }

    protected ZpkFile(AbstractFile file, RandomAccessInputFile input, long length, long lastModified,
                      byte[] checksum) {
        this.file = file;
        this.input = input;
        this.length = length;
        this.lastModified = lastModified;
        this.checksum = checksum;
    }

    public ZpkFile(AbstractFile file) throws IOException {
        this.file = file;
        this.input = RandomAccessInputFile.from(file);
        final long fileSize = input.length();
        RandomAccessByteArrayOutputStream mem = new RandomAccessByteArrayOutputStream(1024);
        mem.order(ByteOrder.LITTLE_ENDIAN);
        ZpkEntry headerEntry = loadEntry(input, mem);
        if (!HEADER_FILENAME.equals(headerEntry.name) || headerEntry.size > 0x7FFFFFFFL) {
            throw new IOException("not a zpk file");
        }
        input.skipBytes((int) headerEntry.size);
        long position = input.getFilePointer();
        long mtime = headerEntry.time;
        for (;;) {
            if (position == fileSize) {
                this.length = fileSize;
                break;
            } else if (position > fileSize) {
                throw new IOException("zpk file end");
            } else if (checkAppendingTag(input)) {
                this.length = position;
                break;
            }
            input.seek(position);
            ZpkEntry entry = loadEntry(input, mem);
            position = input.getFilePointer();
            if (entry.time < mtime) {
                throw new IOException("entry.time: " + sdf1.format(entry.time) +
                        ", header.time: " + sdf1.format(headerEntry.time));
            }
            mtime = entry.time;
            if (entry.name.startsWith(DATA_FILENAME)) {
                String name = entry.name.substring(DATA_FILENAME.length() + 1);
                versions.add(new Version(name, position, entry.size));
            }
            input.seek(position += entry.size);
        }
        if (versions.isEmpty()) {
            throw new IOException("no version");
        }
        MessageDigest md;
        try {
            md = MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException e) {
            throw new IOException(e);
        }
        byte[] m = mem.toByteArray();
        md.update(m, 0, m.length);
        this.checksum = md.digest();
        this.lastModified = mtime;
    }

    public AbstractFile getFile() {
        return file;
    }

    public long length() {
        return length;
    }

    public long getLength() {
        return length;
    }

    public long getLastModified() {
        return lastModified;
    }

    public byte[] getChecksum() {
        return checksum;
    }

    public Version getVersion(String version) {
        for (Version ver : versions) {
            if (version.equals(ver.version)) {
                return ver;
            }
        }
        return null;
    }

    public Version version(String version) throws IOException {
        for (Version ver : versions) {
            if (version.equals(ver.version)) {
                return ver;
            }
        }
        throw new IOException("version not found: " + version);
    }

    public Finder finder() throws IOException {
        if (null == finder) {
            finder = new Finder(versions);
        }
        return finder;
    }

    public RandomAccessOutputFile getOutput() throws IOException {
        RandomAccessOutputFile out = RandomAccessOutputFile.from(file);
        out.order(ByteOrder.LITTLE_ENDIAN);
        out.seek(length);
        return out;
    }

    public static ZpkFile open(String path) throws IOException {
        return new ZpkFile(AbstractFile.make(path));
    }

    public static ZpkFile open(AbstractFile file) throws IOException {
        return new ZpkFile(file);
    }
}
